/*
* Creation date : Tues Mar 03 09:00:00 2007
* Last modified : %modify_time%
*/
/** @file
* \brief This file contains implementation of 
* LLF_ECDH_SVDP_DH function. 
*
* \version LLF_ECDH_SVDP_DH.c#1:csrc:1
* \author Yermalayeu Ihar
* \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
* All Rights reserved
*/

/************************ Include Files ***********************/

#include "LLF_ECPKI_DH.h"
#include "tomcrypt.h"

/************************ Defines *****************************/
/************************ Enums *******************************/
/************************ Typedefs ****************************/
/************************ Global Data *************************/
/************************ Private function prototype **********/
/************************ Private Functions *******************/
/************************ Public Functions ********************/

/**
****************************************************************
* Function Name: 
*  LLF_ECDH_SVDP_DH
*
* Inputs:
* @param BuildedPartnerPublKey_ptr [in] - A pointer to a partner public key. 
* @param UserPrivKey_ptr [in] - A pointer to a user private key.
* @param SharedSecretValue_ptr [out] - A pointer to an output buffer that will 
*                          contain the shared secret value.
* @param SharedSecrValSize_ptr [in\out] - A pointer to the size of user passed 
*                          buffer (in) and actual output size of 
*                          calculated shared secret value.
*
* Outputs:
* @returns \b CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*
* \brief \b 
* Description:
*  Creates the shared secret value accordingly to the IEEE 1363-2000 
*  standard.
*
*  \b 
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_ECDH_SVDP_DH 
*     for creation of shared secret with using LibTomCrypt.
***************************************************************/
CE2Error_t LLF_ECDH_SVDP_DH( 
						CE2_ECPKI_UserPublKey_t *BuildedPartnerPublKey_ptr, /*in*/
						CE2_ECPKI_UserPrivKey_t *UserPrivKey_ptr,           /*in*/							                
						DxUint8_t		            *SharedSecretValue_ptr,     /*out*/
						DxUint32_t              *SharedSecrValSize_ptr    /*in/out*/)
{
  int error_code = CRYPT_OK;
  ecc_key private_key, public_key;
  CE2Error_t result = CE2_OK;

  /* Initialize ltc_mp structure */
  ltc_mp = ltm_desc;

  error_code =  ecc_import(BuildedPartnerPublKey_ptr->PublKeyDbBuff, 
    BuildedPartnerPublKey_ptr->valid_tag, 
    (DomainID_t) BuildedPartnerPublKey_ptr->DomainID, &public_key);
  if (error_code != CRYPT_OK) {
    return CE2_LLF_ECPKI_MODULE_ERROR_BASE;
  }

  error_code =  ecc_import(UserPrivKey_ptr->PrivKeyDbBuff, 
    UserPrivKey_ptr->valid_tag, 
    (DomainID_t) UserPrivKey_ptr->DomainID, &private_key);
  if (error_code != CRYPT_OK) {
    return CE2_LLF_ECPKI_MODULE_ERROR_BASE;
  }

  error_code = ecc_shared_secret(&private_key, &public_key, 
    SharedSecretValue_ptr, SharedSecrValSize_ptr);
  if (error_code != CRYPT_OK) {
    if (error_code == CRYPT_BUFFER_OVERFLOW) {
      result =  LLF_ECPKI_SVDP_DH_INVALID_SHARED_SECRET_VALUE_SIZE_ERROR;
    } else {
      result = CE2_LLF_ECPKI_MODULE_ERROR_BASE;
    }
  }

  ecc_free(&private_key);
  ecc_free(&public_key);
  return result;
}

